// SPDX-License-Identifier: MIT
pragma solidity 0.8.13;

contract NFT {

    // unique values with an owner
    // value => owner
    // we can use a mapping
    mapping(uint => address) public NFTMap; // unique integer => address of the owner
    uint public NFT_value;                  // cost of a NFT (in wei)

    // note: mappings cannot be iterated, i.e. not possible to do (for key in NFTMap.keys())
        // the key space is 2^256, keys of 256 bits, 32 bytes
    mapping(address => uint[]) private NFTList;

    event NFTBought(uint _nft);             // Main purpose for front end applications

    constructor() {
        NFT_value = 1000000000000000;
        // populate the map with a NFTs of a "special" code
        NFTMap[42] = msg.sender;    // address of the account who triggered the transaction to create the contract
        NFTList[msg.sender].push(42);
    }

    function listMyNFTs() public view returns(uint[] memory) {
        return NFTList[msg.sender];
    }

    function listNFTsOf(address _of) public view returns(uint[] memory) {
        return NFTList[_of];
    }

    function buyNFT(uint _nft) public payable returns(bool) {

        // payable means "this function requires "money" >=0 to work"
            // money, the transaction value in fractions of ETH, must be specified in the transaction
        uint money = msg.value;
        uint change;

        // the transaction does not fail
            // does not return funds
            // since this function is payable, this is not secure because the funds are sent to the contract but not managed
                // risk: lose the funds forever, must deal with it with code
        // if(money < NFT_value)
        //     return false;

        // can be written with "require()"
            // revert: the transaction fails
            // failing means that everything is reverted, including the funds sent being sent back to the sender
            // in this case, more secure than return false
        // if(money < NFT_value)
        //     revert("Value too small");

        require(money >= NFT_value, "Value too small");

        NFTMap[_nft] = msg.sender;
        NFTList[msg.sender].push(_nft);
        emit NFTBought(_nft);

        if(money > NFT_value) {
            change = msg.value - NFT_value;
            // Reimbourse the change
            payable(msg.sender).transfer(change);
        }

        return true;
    }

}