package peersim;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import peersim.cdsim.CDProtocol;
import peersim.config.Configuration;
import peersim.core.CommonState;
import peersim.core.Node;

public class Cyclon implements CDProtocol
{
	// This array contains the neighbours of the node
	// It will be updated during computation
	// protected ArrayList<PeerDescriptor> view;
	protected Map<Long,PeerDescriptor> view;
	
	
	/** config parameter name and value for the shuffle length */
	public static final String PAR_SHUFFLE_LENGTH = "shuffle";
	private int shuffle_length;
	
	/** config parameter name and value for the view length */
	public static final String PAR_VIEW_LENGTH = "view";
	private int view_length;
	
	
	
	public Cyclon(String prefix)
	{
		shuffle_length = Configuration.getInt(prefix+"."+PAR_SHUFFLE_LENGTH);
		view_length = Configuration.getInt(prefix+"."+PAR_VIEW_LENGTH);
	}
	
	private void init()
	{
		// view = new ArrayList<PeerDescriptor>(view_length);
		view = new HashMap<Long, PeerDescriptor>();
	}
	
	
	public int getShuffleLength() 
	{
		return shuffle_length;
	}
	

	public int getViewLength() 
	{
		return view_length;
	}
	
	public Map<Long,PeerDescriptor> getView()
	{
		return view;
	}
	
	public void addNeighbour(PeerDescriptor pd)
	{
		view.put(pd.getNode().getID(), pd);
	}
	

	@Override
	public void nextCycle(Node node, int pid) 
	{		
		// 0. increase age of the descriptor in the view
		this.increaseAge();
		
		// 1. choose the peer
		PeerDescriptor pivot = selectPeer();
		Cyclon protocol = (Cyclon) (pivot.getNode().getProtocol(pid));
		
		// 2. selectToSend
		ArrayList<PeerDescriptor> tosend = selectToSend(pivot.getNode(), node);
		
		// 3. do all the sending...
		ArrayList<PeerDescriptor> received = protocol.receive(node, pivot.getNode(), tosend);
		
		// 4. selectToKeep
		selectToKeep(node, tosend, received);
	}
	
	/**
	 * Increase the age of all peer descriptors in the view
	 */
	private void increaseAge()
	{
		for (PeerDescriptor pd: view.values())
			pd.increaseAge();
	}
	
	/**
	 * Gets a random PeerDescriptor from the view
	 * @return
	 */
	private PeerDescriptor selectPeer()
	{
		int index = CommonState.r.nextInt(view.size());
		List<Long> keys = new ArrayList<Long>(view.keySet());
		return view.get(keys.get(index));
	}
	
	
	/**
	 * Selects the descriptors to send to the other peer
	 * @param pivot
	 * @param sender
	 * @return
	 */
	private ArrayList<PeerDescriptor> selectToSend(Node pivot, Node sender)
	{	
		ArrayList<PeerDescriptor> candidates = new ArrayList<PeerDescriptor>(shuffle_length);
		
		for (PeerDescriptor pd: view.values())
		{
			if (pd.getNode().getID() != pivot.getID())
				candidates.add(pd.clone());
		}
		
		// shuffle to randomize
		Collections.shuffle(candidates);
	
		// fill up the data to send
		ArrayList<PeerDescriptor> tosend = new ArrayList<PeerDescriptor>(shuffle_length);
		for (int i=0; i<shuffle_length - 1; i++)
			tosend.add(candidates.get(i));
		
		// add himself to the data
		PeerDescriptor pd = new PeerDescriptor();
		pd.setNode(sender);
		tosend.add(pd);
		
		return tosend;
	}
	
	/**
	 * This method does the following:
	 * 1. removes from received the peers already contained in the view 
	 * 2. free the space for the possible new descriptor
	 * 3. add the new descriptors to the view
	 * @param sent - the peer descriptors sent to the other peer
	 * @param received - the peer descriptors received from the other peer
	 */
	private void selectToKeep(Node myself, ArrayList<PeerDescriptor> sent, ArrayList<PeerDescriptor> received)
	{	
//		System.out.println("------------ I'm node "+myself.getID()+" ---------------");
//		System.out.println("I received "+received.toString());
//		System.out.println("I sent "+sent.toString());
//		
//		System.out.println("My view "+this.toString());
		
		
		// step 1. Remaining peers are added to filtered
		ArrayList<PeerDescriptor> filtered = new ArrayList<PeerDescriptor>(sent.size());
		for (PeerDescriptor pd: received)
		{
			if ((view.containsKey(pd.getNode().getID()) == false) && (myself.getID() != pd.getNode().getID()))
				filtered.add(pd);
		}
		
		
		// this is the amount of space i need to free in the view
		int spaces = filtered.size();
		if (spaces <= 0)
			return;
		
		for (PeerDescriptor pd: sent)
		{
			this.view.remove(pd.getNode().getID());
			
			// if this happens i removed enough
			if (this.view.size() + spaces <= this.view_length)
				break;
		}
		
		//now fill the empty spaces
		for (PeerDescriptor pd: filtered)
		{
			if (this.view.size() >= this.view_length)
				break;
			this.view.put(pd.getNode().getID(), pd);
		}

		// System.out.println("Now my view is: "+this.toString());
	}
	
	// this method is called by another peer to simulate the communication
	public ArrayList<PeerDescriptor> receive(Node caller, Node receiver, ArrayList<PeerDescriptor> received)
	{
		// prepare the data to send back to the caller
		ArrayList<PeerDescriptor> tosend = selectToSend(caller, receiver);
		
		selectToKeep(receiver, tosend, received);
		
		return tosend;
	}


	/**
	 * This is the default mechanism of peersim to create 
	 * copies of objects. To generate a new protocol,
	 * peersim will call this clone method.
	 */
	public Object clone()
	{
		Cyclon p = null;
		try
		{ 
			p = (Cyclon) super.clone();
			p.init();
		}
		catch( CloneNotSupportedException e ) {} // never happens
		return p;
	}
	
	
	public String toString()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("[");
		for (PeerDescriptor pd: view.values())
			sb.append(pd.getNode().getID()).append(",");
		sb.append("]");
		return sb.toString();
			
	}
	
}
