package peersim;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import peersim.cdsim.CDProtocol;
import peersim.config.Configuration;
import peersim.core.Node;

public class Vicinity implements CDProtocol
{
	protected Point2D.Double position;
	
	// This array contains the neighbours of the node
	// It will be updated during computation
	// protected ArrayList<PeerDescriptor> view;
	protected Map<Long,VicinityDescriptor> view;
	
	/** config parameter name and value for the view length */
	public static final String PAR_VIEW_LENGTH = "view";
	private int view_length;
	
	/** config parameter name and value for the shuffle length */
	public static final String PAR_SHUFFLE_LENGTH = "shuffle";
	private int shuffle_length;
	
	/** config parameter for sampling protocol pid **/
	private final String PAR_PROTID = "sampling_protocol";
	private final int sampling_pid;
	
	
	public Vicinity(String prefix)
	{
		shuffle_length = Configuration.getInt(prefix+"."+PAR_SHUFFLE_LENGTH);
		view_length = Configuration.getInt(prefix+"."+PAR_VIEW_LENGTH);
		sampling_pid = Configuration.getPid(prefix+"."+PAR_PROTID);
	}
	
	private void init()
	{ 
		view = new HashMap<Long, VicinityDescriptor>();
	}
	
	@Override
	public void nextCycle(Node node, int pid) 
	{	
		//System.out.println();
		//System.out.println("Next cycle of: "+node.getID());
		
		// 1. choose the peer and take its vicinity protocol
		VicinityDescriptor pivot = selectPeer(node);
		Vicinity vicinity = (Vicinity) (pivot.getNode().getProtocol(pid));
		
		//System.out.println("Contacting "+pivot.getNode().getID());
		
		// 1.1 choose my own cyclon protocol
		Cyclon cyclon = (Cyclon) (node.getProtocol(sampling_pid));
		
		// 2. selectToSend
		//System.out.println("Calling selectToSend");
		List<VicinityDescriptor> tosend = selectToSend(pivot, new ArrayList<VicinityDescriptor>(cyclon.getView().values()));
		
		// 3. do all the sending...
		VicinityDescriptor me = new VicinityDescriptor();
		me.setNode(node);
		me.setPosition(this.position);
		List<VicinityDescriptor> received = vicinity.receive(me, pivot, tosend);
		
		// 4. selectToKeep
		selectToKeep(received, new ArrayList<VicinityDescriptor>(cyclon.getView().values()));
		
	}
	
	/**
	 * Merge the data received from the other peer with
	 * the the local views from Vicinity and from Cyclon.
	 * Then keeps the semantically closer peers among all. 
	 */
	private void selectToKeep(List<VicinityDescriptor> received, List<VicinityDescriptor> cyclon)
	{
		Map<Long,VicinityDescriptor> merged = new HashMap<Long,VicinityDescriptor>();
		
		// Merge received, cyclon and view descriptor in a single data structure
		// If the same node is inserted multiple time, the freshest is kept.
		merge(merged, received);
		merge(merged, cyclon);
		merge(merged, new ArrayList<VicinityDescriptor>(this.view.values()));
		
		// rank semantically accordng to myself
		List<VicinityDescriptor> values = new ArrayList<VicinityDescriptor>(merged.values());
		Collections.sort(values, new VicinityComparator(this.position));
		
		// update the local view
		this.view.clear();
		for (int i=0; i<view_length; i++)
		{
			this.view.put(values.get(i).getNodeId(), values.get(i));
		}
		// System.out.println(">> "+view.values());
	}
	
	/**
	 * Gets the PeerDescriptor with the oldest timestamp
	 * @return
	 */
	private VicinityDescriptor selectPeer(Node node)
	{
		List<VicinityDescriptor> touse = null; 
		
		if (view.isEmpty())
		{
			Cyclon cyclon = (Cyclon) (node.getProtocol(sampling_pid));
			touse = new ArrayList<VicinityDescriptor>(cyclon.getView().values());
		}
		else
			touse = new ArrayList<VicinityDescriptor>(view.values());
		
		Collections.sort(touse);
		return touse.get(touse.size() - 1);
	}
	
	/**
	 * Selects the descriptors to send to the other peer
	 * @param pivot
	 * @param sender
	 * @return
	 */
	private List<VicinityDescriptor> selectToSend(VicinityDescriptor pivot, List<VicinityDescriptor> cyclon)
	{	
		Map<Long,VicinityDescriptor> merged = new HashMap<Long,VicinityDescriptor>();
		
//		System.out.println("Content of cyclon");
//		for (VicinityDescriptor vd: cyclon)
//		{
//			System.out.print("\t"+vd.getNodeId()); 
//			System.out.println(" "+vd.getPosition());
//		}
//		
//		System.out.println("Content of view");
//		for (VicinityDescriptor vd: view.values())
//		{
//			System.out.print("\t"+vd.getNodeId()); 
//			System.out.println(" "+vd.getPosition());
//		}
		
		
		// Merge cyclon and view descriptor in a single data structure
		// If the same node is inserted multiple times, the freshest is kept.
		merge(merged, cyclon);
		merge(merged, new ArrayList<VicinityDescriptor>(this.view.values()));
		
		
		// rank semantically ***according to the receiver***
		List<VicinityDescriptor> values = new ArrayList<VicinityDescriptor>(merged.values());
//		System.out.println("Content of merged");
//		for (VicinityDescriptor vd: values)
//		{
//			System.out.print("\t"+vd.getNodeId()); 
//			System.out.println(" "+vd.getPosition());
//		}
		Collections.sort(values, new VicinityComparator(pivot.getPosition()));
		
		// prepare the data structure to send
		int send_lenght = Math.min(values.size(), this.shuffle_length);
		
		List<VicinityDescriptor> tosend = new ArrayList<VicinityDescriptor>(send_lenght);
		for (int i=0; i<send_lenght; i++)
			tosend.add(values.get(i).clone());
		
		return tosend;
	}
	
	
	// this method is called by another peer to simulate the communication
	public List<VicinityDescriptor> receive(VicinityDescriptor caller, VicinityDescriptor receiver, List<VicinityDescriptor> sent)
	{
		// System.out.println("Node "+receiver.getNodeId()+" receive from "+caller.getNodeId());
		// who does the receive take its cyclon
		Cyclon cyclon = (Cyclon) (receiver.getNode().getProtocol(sampling_pid));
		
		// prepare the data to send back to the caller
		List<VicinityDescriptor> tosend = this.selectToSend(caller, new ArrayList<VicinityDescriptor>(cyclon.getView().values()));
		
		// keep the data
		this.selectToKeep(sent, new ArrayList<VicinityDescriptor>(cyclon.getView().values()));
			
		return tosend;
	}
	
		
	
	/**
	 * Merges the given descriptor inside the merged Map
	 * @param merged
	 * @param tomerge
	 */
	private void merge(Map<Long,VicinityDescriptor> merged, List<VicinityDescriptor> tomerge)
	{
		// Add the nodes from cyclon
		for (VicinityDescriptor vd: tomerge)
		{
			VicinityDescriptor inside = merged.get(vd.getNodeId());
			if (inside != null) // they have the same id, keep the freshest
			{
				if (inside.compareTo(vd) > 0) // this means vd has lower age
					merged.put(vd.getNodeId(), vd);
			}
			else
				merged.put(vd.getNodeId(), vd);
		}
	}
	
	/**
	 * This is the default mechanism of peersim to create 
	 * copies of objects. To generate a new protocol,
	 * peersim will call this clone method.
	 */
	public Object clone()
	{
		Vicinity p = null;
		try
		{ 
			p = (Vicinity) super.clone();
			p.init();
			p.position = this.position;
		}
		catch( CloneNotSupportedException e ) {} // never happens
		return p;
	}
	
	
	@Override
	public String toString()
	{
		return this.position + " "  +view.toString();
	}
	
	class VicinityComparator implements Comparator<VicinityDescriptor>
	{
		private Point2D.Double _position;
		
		public VicinityComparator(Point2D.Double pos)
		{
			this._position = pos;
		}
		
		@Override
		public int compare(VicinityDescriptor v1, VicinityDescriptor v2) 
		{
			// System.out.println("Comparing "+v1+ " "+v2);
			
			double d1 = v1.getPosition().distance(this._position);
			double d2 = v2.getPosition().distance(this._position);
			
			if (d1 < d2)
				return -1;
			else if (d1 > d2)
				return 1;
			else return 0;
		}
		
	}
	
}
