package peersim;

import java.util.ArrayList;

import peersim.cdsim.CDProtocol;
import peersim.core.CommonState;
import peersim.core.Linkable;
import peersim.core.Node;

/**
 * This class implements a simple gossip protocols that computes the average
 * of the values in a network. 
 * Implements:
 * CDProtocol - the peersim interface that define a cycle driven protocol
 * Linkable - the peersim interface that defines the ability of being in a topology 
 * @author carlini
 *
 */
public class Average implements CDProtocol, Linkable
{
	// This array contains the neighbours of the node 
	// All the arrays of all the nodes define the topology of the network.
	// It is fixed during all the computation
	protected ArrayList<Node> neighbours;
	
	// This is the value of the node. It is initialised at the start
	// and refined incrementally by the protocol. 
	protected double value;
	
	/**
	 * Constructor. It is empty as our protocol is very simple and 
	 * does not need any initialization
	 * @param prefix
	 */
	public Average(String prefix)
	{
	}
	
	/**
	 * Return the value
	 * @return
	 */
	public double getValue()
	{
		return value;
	}
	 
	/**
	 * Set the value
	 * @param value
	 */
	public void setValue(double value)
	{
		this.value = value;
	}
	
	/**
	 * This is the default mechanism of peersim to create 
	 * copies of the objects. To generate a new average protocol,
	 * peersim will call this clone method.
	 */
	public Object clone()
	{
		Average af = null;
		try
		{ 
			af = (Average) super.clone();
			af.neighbours = new ArrayList<Node>();
		}
		catch( CloneNotSupportedException e ) {} // never happens
		return af;
	}
	
	
	// ---- OVERRIDE from CDProtocol 
	
	/**
	 * The business code of the protocol.
	 * A typical protocol iteration is (i) to read some data 
	 * (neighbor.value in this case), (ii) to modify the
	 * internal state, and (iii) to "send" out the modified state 
	 */
	@Override
	public void nextCycle(Node node, int pid) 
	{
		Linkable linkable = (Linkable) node.getProtocol(pid);
		if (linkable.degree() > 0)
		{
			Node peer = linkable.getNeighbor(CommonState.r.nextInt(linkable.degree()));
			
			Average neighbor = (Average) peer.getProtocol(pid);
			double mean = (this.value + neighbor.value) / 2;
			this.value = mean; // modify internal state
			neighbor.value = mean; // send out the modified state
		}

	}
	
	
	// ---- OVERRIDE from Linkable
	/* Basic management of the neighbours.
	 * These methods are called by the WireKOut initializer
	 * to create the initial (and final) topology
	 */
	

	@Override
	public void onKill() {
		neighbours = null;
	}

	@Override
	public int degree()
	{
		return neighbours.size();
	}

	@Override
	public Node getNeighbor(int i) 
	{
		return neighbours.get(i);
	}

	/**
	 * Add a node only if the node is not already present
	 */
	@Override
	public boolean addNeighbor(Node neighbour) 
	{
		if (neighbours.contains(neighbour))
			return false;
		
		neighbours.add(neighbour);
		return true;
	}

	@Override
	public boolean contains(Node neighbor) 
	{
		return neighbours.contains(neighbor);
	}

	@Override
	public void pack() {
		System.out.println("Average.pack() -- not implemented");
	}

}
